/**
 * 
 */
package gov.va.med.mhv.usermgmt.service;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Date;

import junit.framework.Assert;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.service.ServiceResponse;

import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.service.delegate.UserProfileServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.UserRegistrationServiceDelegate;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;

/**
 * Utility class to support user testing.
 * Must call setUp() to register the user. A user with a random user account, 
 * if no user name is specified in the given UserProfile.
 * Must call tearDown() to remove the user.
 * @author Rob Proper
 *
 */
public final class UserBuilder {
	
	private static final Log LOG = LogFactory.getLog(UserBuilder.class);
	
	UserProfile userProfile;
	final String password;
	final boolean isIPAed;

	public static String generateUserName() {
		int currentYear = Calendar.getInstance().get(Calendar.YEAR);
		long epoch = createDate(Calendar.JANUARY, 1, currentYear).getTime();
		long elapsed = Calendar.getInstance().getTimeInMillis() - epoch;
		String userName = Long.toHexString(elapsed);
		if (userName.length() < 6) {
			StringUtils.leftPad(userName, 6 - userName.length(), '0');
		}
		return userName;
	}

	public static Date createDate(int month, int day, int year) {
		java.util.Calendar calendar = java.util.Calendar.getInstance();
		calendar.set(Calendar.YEAR, year);
		calendar.set(Calendar.MONTH, month); // Months are 0-based
		calendar.set(Calendar.DAY_OF_MONTH, day);
		calendar.set(Calendar.HOUR, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0); 
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}

	public UserBuilder(UserProfile userProfile, String password) {
		this(userProfile, password, false);
	}
	
	public UserBuilder(UserProfile userProfile, String password, 
		boolean isIPAed) 
	{
		this.userProfile = userProfile;
		this.password = password;
		this.isIPAed = isIPAed;
		// if isIPAed, check user is patient
	}
	
	public void setUp() throws Exception {
		if (StringUtils.isBlank(getUserName())) {
			userProfile.setUserName(generateUserName());
		} else {
			UserProfile profile = getUserProfile();
		    if (find() != null) {
			   LOG.warn("User '" + getUserName() + "' already exists. Tearing"
				   + " down old user before setting up new one.");
			   tearDown();
			   // use original profile
			   this.setUserProfile(profile);
		    }
		}
		UserRegistrationServiceDelegate delegate = 
			ServiceDelegateFactory.createUserRegistrationServiceDelegate();
		UserProfileServiceResponse response = delegate.registerUser(
			getUserProfile(), getUserProfile().getSsn(), getUserName(), 
			getPassword(), getPassword(), new Timestamp(Calendar.getInstance().
			getTimeInMillis()));
		Assert.assertFalse("Unable to register user '" + getUserName() + "': "
			+ response.getMessages().getErrorMessages(), response.getMessages().
			hasErrorMessages());
		Assert.assertFalse("Unable to register user '" + getUserName() + "': "
			+ response.getMessages().getErrorMessages(), response.getMessages().
			hasErrorMessages());
		setUserProfile(response.getUserProfile());
		if (isIPAed()) {
			// Add code to IPA user
		}
	}

	public void tearDown() throws Exception {
		if (!StringUtils.isBlank(getUserName())) {
			ServiceResponse response = ServiceDelegateFactory.
				createUserServiceDelegate().removeUser(getUserName());
			if (response.getMessages().hasErrorMessages()) {
				LOG.warn("Problems removing user'" + getUserName() + "': "
					+ response.getMessages().getErrorMessages());
			}
		}
		if (getUserProfile() != null) {
			ServiceResponse response = ServiceDelegateFactory.
				createEntityMaintenanceServiceDelegate().delete(
				getUserProfile().getUserProfilePK());
			if (response.getMessages().hasErrorMessages()) {
				LOG.warn("Problems deleting user profile for user '" 
					+ getUserName() + "': " + response.getMessages().
					getErrorMessages());
			}
		}
		userProfile = null;
	}
	
	public UserProfile getUserProfile() { 
		return userProfile; 
	}
	public void setUserProfile(UserProfile  userProfile) { 
		this.userProfile = userProfile; 
	}
	public String getPassword() { 
		return password; 
	}
	public String createInvalidPassword() { 
		return (StringUtils.isBlank(password)) ? "dontcare" 
			: ((password.length() > 1) ? password.substring(1) : "dontcare"); 
	}
	public String createResetPassword() { 
		String password = "%dontcare0";
		if(!StringUtils.isBlank(password) && (password.length() > 1)) {
			char ch = password.charAt(0);
			if (Character.isDigit(ch)) {
				ch = (ch == '0') ? '1' : '0';
			} else if (Character.isLetter(ch)) {
				ch = (ch == 'a') ? 'b' : 'a'; 
			} else {
				ch = (ch == '%') ? '$' : '%';
			}
			password = "" + ch + password.substring(1); 
		}
		return password;
	}
	public String getUserName() { 
		return (userProfile != null) ? userProfile.getUserName() : null; 
	}
	public boolean isIPAed() { 
		return isIPAed; 
	}

	public UserProfile find() {
		UserProfileServiceDelegate delegate = ServiceDelegateFactory.
			createUserProfileServiceDelegate();
		UserProfileServiceResponse response = delegate.getProfileForUser(
			getUserName());
		if (response.getMessages().hasErrorMessages()) {
			LOG.error("Problems finding user '" + getUserName() + "'"
				+ response.getMessages().getErrorMessages());
		}
		return response.getUserProfile();
	}

}